<?php
namespace App\Repositories;

use App\Interfaces\UnitRepositoryInterface;
use App\Models\Unit;
use Illuminate\Support\Facades\DB;
use PermissionHelper;

class UnitRepository implements UnitRepositoryInterface
{
    protected $model;

    public function __construct(Unit $unit)
    {
        $this->model = $unit;
    }

    public function all()
    {
        $user = auth()->user();

        $query = $this->model
            ->whereHas('location', function ($q) {
                $q->whereNull('deleted_at');
            });

        if (!PermissionHelper::hasAllAccess()) {
            $query->where('location_id', $user->location_id);
        }

        return $query
            ->orderBy('location_id', 'asc')
            ->FilterByRole()
            ->orderBy('id', 'desc')
            ->get();
    }


    public function getDashboard()
    {
        $unitStats = DB::table('unit as e')
            ->join('type_units as t', 'e.type_id', '=', 't.id')
            ->select('t.name as type_name', DB::raw('COUNT(e.id) as total_unit'))
            ->groupBy('t.id', 't.name')
            ->orderByDesc('total_unit')
            ->get();

        return $unitStats;
    }

    public function datatableMonthly()
    {
        $user = auth()->user();

        if (!PermissionHelper::hasAllAccess()) {
            return $this->model->where('location_id', $user->location_id)
                ->when(empty(request('order')) || (request('order')[0]['name'] ?? 'DT_RowIndex') === 'DT_RowIndex', fn($q) => $q->orderBy('units.id', 'desc'))
                // ->orderBy('id', 'desc')
            ;
        }

        return $this->model
            ->when(empty(request('order')) || (request('order')[0]['name'] ?? 'DT_RowIndex') === 'DT_RowIndex', fn($q) => $q->orderBy('units.id', 'desc'))
            // ->orderBy('id', 'desc')
        ;
    }

    public function find($id)
    {
        return $this->model->findOrFail($id);
    }

    public function create(array $data)
    {
        return $this->model->create($data);
    }

    public function update($id, array $data)
    {
        $unit = $this->find($id);
        $unit->update($data);
        return $unit;
    }

    public function delete($id)
    {
        $unit = $this->find($id);
        return $unit->delete();
    }

    public function datatable()
    {
        $user = auth()->user();

        $query = $this->model
            ->with('location')
            ->whereHas('location', function ($q) {
                $q->whereNull('deleted_at');
            })
            ->when(empty(request('order')) || (request('order')[0]['name'] ?? 'DT_RowIndex') === 'DT_RowIndex', fn($q) => $q->orderBy('units.id', 'desc'))
            // ->orderBy('id', 'desc')
        ;

        if (!PermissionHelper::hasAllAccess()) {
            $query->where('location_id', $user->location_id);
        }

        return $query->filterByRole();
    }


}
