<?php
namespace App\Repositories;

use App\Interfaces\DailyRepositoryInterface;
use App\Models\Daily;
use App\Models\DetailDaily;
use Illuminate\Support\Facades\File;
use PermissionHelper;

class DailyRepository implements DailyRepositoryInterface
{
    private $daily;

    public function __construct(Daily $daily)
    {
        $this->daily = $daily;
    }

    public function all()
    {
        $isAdmin = PermissionHelper::hasAllAccess();
        $query = $this->daily->orderBy("id", "desc");
        if (!$isAdmin) {
            $query->where("location_id", auth()->user()->location_id);
        }

        return $query->filterByRole()->get();
    }

    public function datatable(
        ?int $sitesId = null,
        ?int $shiftId = null,
        ?string $date = null
    ) {
        $isAdmin = PermissionHelper::hasAllAccess();

        $query = $this->daily
            ->with(['form', 'location'])
            ->whereHas('location', function ($q) {
                $q->whereNull('deleted_at');
            })
            ->when($sitesId, fn($q) => $q->where('location_id', $sitesId))
            ->when($shiftId, fn($q) => $q->where('shift_id', $shiftId))
            ->when($date, fn($q) => $q->whereDate('date', $date))
            ->when(empty(request('order')) || (request('order')[0]['name'] ?? 'DT_RowIndex') === 'DT_RowIndex', fn($q) => $q->orderBy('dailies.id', 'desc'))
            // ->orderBy('id', 'desc')
        ;

        if (!$isAdmin) {
            $query->where('location_id', auth()->user()->location_id);
        }

        $isDriverOperator = in_array(strtolower(auth()->user()->role->role), ["driver", "operator"]);

        // if (!$allAccess && $userId !== null) {
        //     $query->where('dailies.user_id', $userId);
        // }
        if ($isDriverOperator) {
            $query->where('dailies.user_id', auth()->user()->id);
        }
        return $query->filterByRole();
    }


    public function find($id)
    {
        return $this->daily->findOrFail($id);
    }

    public function create(array $data)
    {
        return $this->daily->create($data);
    }

    public function update($id, array $data)
    {
        $location = $this->find($id);
        $location->update($data);
        return $location;
    }

    public function delete($id)
    {
        $daily = $this->daily->find($id);
        $year = \Carbon\Carbon::parse($daily->date)->format('Y');
        $month = \Carbon\Carbon::parse($daily->date)->format('m');
        $locationName = str_replace(' ', '_', $daily->location->name);

        $folderPath = public_path("PDF_PMM/{$year}/{$month}/{$locationName}/{$daily->shift->name}");



        $fileName = "{$daily->code_report}.pdf";
        $filePath = "{$folderPath}/{$fileName}";
        if (File::exists($filePath)) {
            File::delete($filePath);
        }

        DetailDaily::where("dailies_id", $id)->delete();
        return $daily->delete();

    }
}
