<?php

namespace App\Http\Controllers;

use App\Interfaces\LocationRepositoryInterface;
use App\Interfaces\RoleRepositoryInterface;
use App\Interfaces\UserRepositoryInterface;
use App\Models\User;
use Illuminate\Http\Request;
use PermissionHelper;
use Yajra\DataTables\DataTables;

class UserController extends Controller
{

    protected $userRepository;
    protected $rolesRepository;
    protected $locationRepository;

    public function __construct(UserRepositoryInterface $userRepository, RoleRepositoryInterface $rolesRepository, LocationRepositoryInterface $locationRepository)
    {
        $this->userRepository = $userRepository;
        $this->rolesRepository = $rolesRepository;
        $this->locationRepository = $locationRepository;
    }
    public function index()
    {
        $users = $this->userRepository->datatable();
        // dd($users->where('role_id', '1')
        //     ->first());
        $usersTrash = $this->userRepository->datatableTrash();
        $roles = $this->rolesRepository->all();
        $location = $this->locationRepository->all();
        $canCreate = PermissionHelper::hasPermission("users")->create;
        if (request()->ajax()) {
            return $this->datatable(request()->option == "active" ? $users : $usersTrash);
        }
        return view("master-data.users.index", compact('roles', 'location', "canCreate"));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {

    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = request()->validate([
            "name" => "required",
            "tgl_lahir" => "required",
            "nik" => "required",
            "no_hp" => "required",
            "email" => "required|email|unique:users,email", // disarankan untuk mencegah duplikasi
            "username" => "required|unique:users,username", // disarankan untuk mencegah duplikasi
            "password" => "required|min:8",
            "role_id" => "required",
            "location_id" => "required",
        ]);

        // Enkripsi password sebelum disimpan
        $validated['password'] = bcrypt($validated['password']);
        $validated['approve'] = 1;

        // Simpan user baru
        $user = $this->userRepository->create($validated);

        return response()->json(['message' => 'User berhasil ditambahkan']);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, int $id)
    {
        // Ambil user yang ingin diupdate
        $user = $this->userRepository->find($id);
        if (!$user) {
            return redirect()->back()->withErrors("User tidak ditemukan.");
        }

        // Validasi input
        $validated = $request->validate([
            "name" => "required",
            "tgl_lahir" => "required",
            "nik" => "required",
            "no_hp" => "required",
            "email" => "required|email|unique:users,email,{$id}", // ignore user sendiri
            "username" => "required|unique:users,username,{$id}", // disarankan untuk mencegah duplikasi
            "password" => "nullable|min:8", // opsional
            "role_id" => "required",
            "location_id" => "required",
        ]);

        // Jika password dikirim, enkripsi
        if (!empty($validated['password'])) {
            $validated['password'] = bcrypt($validated['password']);
        } else {
            unset($validated['password']); // jangan ubah jika kosong
        }

        $this->userRepository->update($id, $validated);
        return response()->json(['message' => 'User berhasil diupdate']);
    }


    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            $this->userRepository->delete($id);

            $option = request()->option;
            
            $message = match ($option) {
                "restore" => "User berhasil dipulihkan.",
                "force-delete" => "User dan semua data terkait berhasil dihapus total.",
                default => "User berhasil dihapus sementara."
            };

            return response()->json(['message' => $message]);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Gagal menghapus data: ' . $e->getMessage()], 500);
        }
    }

    public function datatable($data)
    {
        return DataTables::of($data)
            ->addIndexColumn()
            ->addColumn('action', function ($row) {
                $canEdit = PermissionHelper::hasPermission("users")->update && request()->option == "active";
                $canDelete = PermissionHelper::hasPermission("users")->delete;
                if (!$canEdit && !$canDelete) {
                    return "<span class='text-muted fst-italic'>Only View</span>";
                }
                $actionBtn = "<div class='d-flex  gap-1'>";
                if ($canEdit) {
                    $actionBtn .= '
                        <a href="javascript:void(0)"
                            data-id="' . $row->id . '"
                            data-name="' . e($row->name ?? '') . '"
                            data-tgl_lahir="' . e($row->tgl_lahir ?? '') . '"
                            data-nik="' . e($row->nik ?? '') . '"
                            data-no_hp="' . e($row->no_hp ?? '') . '"
                            data-email="' . e($row->email ?? '') . '"
                            data-location="' . e($row->location_id ?? '') . '"
                            data-roles="' . e($row->role->id ?? '') . '"
                            data-username="' . e($row->username ?? '') . '"
                            class="edit btn btn-warning btn-sm">
                            <i class="fa fa-pencil"></i>
                        </a>';
                }

                if ($canDelete && stripos($row->name, 'demo') === false) {
                    if (request()->option == "active") {
                        // Soft Delete
                        $actionBtn .= '
                            <a href="javascript:void(0)"
                                data-id="' . $row->id . '"
                                data-option="delete"
                                class="delete btn btn-danger btn-sm">
                                <i class="fa fa-trash"></i>
                            </a>';
                    } elseif (request()->option == "trash") {
                        // Restore
                        $actionBtn .= '
                            <a href="javascript:void(0)"
                                data-id="' . $row->id . '"
                                data-option="restore"
                                class="delete btn btn-warning btn-sm">
                                <i class="fa fa-trash-arrow-up"></i>
                            </a>';
                        // Force Delete
                        $actionBtn .= '
                            <a href="javascript:void(0)"
                                data-id="' . $row->id . '"
                                data-option="force-delete"
                                class="delete btn btn-dark btn-sm">
                                <i class="fa fa-times"></i>
                            </a>';
                    }
                }


                $actionBtn .= "</div>";

                return $actionBtn;
            })

            ->addColumn("role", function ($row) {
                return $row->role->role ?? "-";
            })
            ->addColumn("sites", function ($row) {
                return $row->location->name;
            })
            ->filterColumn('role', function ($query, $keyword) {
                $query->whereHas('role', fn($q) => $q->where('role', 'like', "%{$keyword}%"));
            })
            ->filterColumn('sites', function ($query, $keyword) {
                $query->whereHas('location', fn($q) => $q->where('name', 'like', "%{$keyword}%"));
            })
            ->orderColumn('role', function ($query, $order) {
                $query->join('roles', 'roles.id', '=', 'users.role_id')
                    ->select('users.*', 'roles.role as role_name') // kasih alias lain
                    ->orderBy('roles.role', $order);
                ;
            })
            ->orderColumn('sites', function ($query, $order) {
                $query->join('locations', 'locations.id', '=', 'users.location_id')
                    ->orderBy('locations.name', $order);
            })
            ->rawColumns(['action'])
            ->make(true);
    }
}