<?php

namespace App\Http\Controllers;

use App\Interfaces\ActionRoleRepositoryInterface;
use App\Interfaces\RoleRepositoryInterface;
use App\Models\Action;
use Illuminate\Http\Request;
use Yajra\DataTables\Facades\DataTables;

class RolesController extends Controller
{
    protected $role;
    protected $actionrole;

    public function __construct(RoleRepositoryInterface $role, ActionRoleRepositoryInterface $actionrole)
    {
        $this->role = $role;
        $this->actionrole = $actionrole;
    }

    public function index()
    {
        $option = request()->option;
        $actions = Action::all();
        if (request()->ajax()) {
            return $this->datatable($option == "active" ? $this->role->all() : $this->role->trashed());
        }
        return view("master-data.role.index", compact("actions"));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'role' => 'required|string|max:100',
            "reads" => "array",
            "updates" => "array",
            "creates" => "array",
            "deletes" => "array",
        ]);
        $data = [
            "role" => $request->role,
            'isAllLocation' => $request->has('all_site') ? 1 : 0,
            'isOperator' => $request->has('is_operator') ? 1 : 0,
        ];
        $result = $this->role->create($data);
        $allActions = collect($request->reads)
            ->merge($request->creates)
            ->merge($request->updates)
            ->merge($request->deletes)
            ->unique();

        $pivotData = [];

        foreach ($allActions as $actionId) {
            $pivotData[$actionId] = [
                "read" => in_array($actionId, $request->reads ?? []),
                "create" => in_array($actionId, $request->creates ?? []),
                "update" => in_array($actionId, $request->updates ?? []),
                "delete" => in_array($actionId, $request->deletes ?? []),
            ];
        }

        // attach dengan kolom tambahan
        $result->actions()->attach($pivotData);
        return response()->json(['message' => 'Role berhasil ditambahkan.']);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'role' => 'required|string|max:100',
            "reads" => "array",
            "updates" => "array",
            "creates" => "array",
            "deletes" => "array",
        ]);

        $role = $this->role->update($id, [
            "role" => $request->role,
            'isAllLocation' => $request->has('all_site') ? 1 : 0,
            'isOperator' => $request->has('is_operator') ? 1 : 0,
        ]);


        $allActions = collect($request->reads)
            ->merge($request->creates)
            ->merge($request->updates)
            ->merge($request->deletes)
            ->unique();

        $pivotData = [];

        foreach ($allActions as $actionId) {
            $pivotData[$actionId] = [
                "read" => in_array($actionId, $request->reads ?? []),
                "create" => in_array($actionId, $request->creates ?? []),
                "update" => in_array($actionId, $request->updates ?? []),
                "delete" => in_array($actionId, $request->deletes ?? []),
            ];
        }

        // sync: hapus relasi lama & isi ulang dengan pivot baru
        $role->actions()->sync($pivotData);

        return response()->json(['message' => 'Role berhasil diupdate.']);
    }


    public function getRoleAkses($id)
    {
        return response()->json($this->role->find($id));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $this->actionrole->deleteByRoleId($id);

        $this->role->delete($id);
        return response()->json(data: ['message' => request()->option == "restore" ? "Role berhasil dipulihkan" : 'Role berhasil dihapus.']);
    }

    public function datatable($data)
    {
        return DataTables::of($data)
            ->addIndexColumn()
            ->addColumn('all_site', function ($row) {
                if ($row->isAllLocation == 1) {
                    return '<i class="fa fa-check text-success"></i>';
                } else {
                    return '-';
                }
            })
            ->addColumn('is_Operator', function ($row) {
                if ($row->isOperator == 1) {
                    return '<i class="fa fa-check text-success"></i>';
                } else {
                    return '-';
                }
            })
            ->addColumn('action', function ($row) {
                $actionBtn = "<div class='d-flex gap-2'>";
                $actionBtn .= request()->option == "active" ? '<a href="javascript:void(0)"  
                    data-id="' . $row->id . '"  
                    data-role="' . $row->role . '" 
                    data-all-site="' . $row->isAllLocation . '" 
                    data-is-Operator="' . $row->isOperator . '" 
                    class="edit btn btn-warning btn-sm"><i class="fa fa-pencil"></i></a>' : "";

                if ($row->id != 1 && request()->option == "active") {
                    $actionBtn .= '<a href="javascript:void(0)"
                    data-id="' . $row->id . '"
                    data-option="' . (request()->option == "active" ? "delete" : "restore") . '"
                     class="delete btn btn-danger btn-sm"><i class="fa fa-trash"/></i></a>';
                }
                if ($row->id != 1 && request()->option == "trash") {
                    $actionBtn .= '<a href="javascript:void(0)"
                    data-id="' . $row->id . '"
                    data-option="' . (request()->option == "active" ? "delete" : "restore") . '"
                     class="delete btn btn-warning btn-sm" title="restore data"><i class="fa fa-trash-arrow-up"/></i></a>';
                }
                $actionBtn .= "</div>";
                return $actionBtn;
            })
            ->rawColumns(['action', 'all_site', 'is_Operator'])
            ->make(true);
    }
}