<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Interfaces\LocationRepositoryInterface;
use App\Interfaces\SavingRepositoryInterface;
use Yajra\DataTables\DataTables;
use Illuminate\Database\QueryException;
use PermissionHelper;

class LocationController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    protected $locationRepository;

    public function __construct(LocationRepositoryInterface $locationRepository)
    {
        $this->locationRepository = $locationRepository;
    }

    public function index()
    {
        $data = $this->locationRepository->datatable();
        $dataTrash = $this->locationRepository->datatableTrash();
        $canCreate = PermissionHelper::hasPermission("locations")->read;

        if (request()->ajax()) {
            return $this->datatable(request()->option == "active" ? $data : $dataTrash);
        }

        return view('master-data.location.index', ['canCreate' => $canCreate]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
        ]);

        try {
            $location = $this->locationRepository->create($validated);

            return response()->json([
                'status' => 'success',
                'message' => 'Lokasi berhasil ditambahkan.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Gagal menambahkan lokasi: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
        ]);

        try {
            $this->locationRepository->update($id, $validated);

            return response()->json([
                'status' => 'success',
                'message' => 'Lokasi berhasil diupdate.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Gagal mengupdate lokasi: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            $this->locationRepository->delete($id);

            return response()->json([
                'success' => true,
                'message' => 'Lokasi berhasil dihapus!'
            ]);
        } catch (QueryException $e) {
            if ($e->getCode() == '23000') {
                return response()->json([
                    'success' => false,
                    'message' => 'Gagal menghapus lokasi karena masih digunakan di data lain.'
                ], 400);
            }

            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan saat menghapus lokasi.'
            ], 500);
        }
    }

    public function datatable($data)
    {
        return DataTables::of($data)
            ->addIndexColumn()
            ->addColumn('action', function ($row) {
                $canEdit = PermissionHelper::hasPermission("locations")->update && request()->option == "active";
                $canDelete = PermissionHelper::hasPermission("locations")->delete;

                if (!$canEdit && !$canDelete) {
                    return "<span class='text-muted fst-italic'>Only View</span>";
                }

                $actionBtn = "<div class='gap-2 d-flex'>";

                if ($canEdit) {
                    $actionBtn .= '
                        <a href="javascript:void(0)"
                            class="edit btn btn-warning btn-sm"
                            data-id="' . $row->id . '"
                            data-name="' . e($row->name ?? '') . '">
                            <i class="fa fa-pencil"></i>
                        </a>';
                }

                if ($canDelete) {
                    if (request()->option == "active") {
                        // Soft Delete
                        $actionBtn .= '
                            <a href="javascript:void(0)"
                                class="delete btn btn-danger btn-sm"
                                data-id="' . $row->id . '"
                                data-option="delete">
                                <i class="fa fa-trash"></i>
                            </a>';
                    } elseif (request()->option == "trash") {
                        // Restore
                        $actionBtn .= '
                            <a href="javascript:void(0)"
                                class="delete btn btn-warning btn-sm"
                                data-id="' . $row->id . '"
                                data-option="restore">
                                <i class="fa fa-trash-arrow-up"></i>
                            </a>';

                        // Force Delete
                        $actionBtn .= '
                            <a href="javascript:void(0)"
                                class="delete btn btn-dark btn-sm"
                                data-id="' . $row->id . '"
                                data-option="force-delete">
                                <i class="fa fa-times"></i>
                            </a>';
                    }
                }

                $actionBtn .= "</div>";
                return $actionBtn;
            })
            ->rawColumns(['action'])
            ->make(true);
    }

}