<?php

namespace App\Http\Controllers;

use App\Interfaces\LocationRepositoryInterface;
use App\Interfaces\UnitRepositoryInterface;
use App\Models\Daily;
use App\Models\Location;
use App\Models\Unit;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    protected $unit;
    protected $location;

    public function __construct(
        UnitRepositoryInterface $unit,
        LocationRepositoryInterface $location,
    ) {
        $this->unit = $unit;
        $this->location = $location;
    }

    public function index()
    {
        $equipmentStats = Unit::join('type_units as t', 'units.type_id', '=', 't.id')
            ->select('t.name as type_name', DB::raw('COUNT(units.id) as total_equipment'))
            ->groupBy('t.id', 't.name')
            ->orderByDesc('total_equipment')
            ->get();

        $sites = $this->location->all();
        $tanggal = request()->tanggal ?? now()->toDateString();

        $dailyList = Daily::with(['shift', 'unit', 'detail'])
            ->select("dailies.unit_id", "dailies.shift_id", "dailies.id")
            ->whereDate('date', $tanggal)
            ->groupBy("dailies.id", 'dailies.unit_id', 'dailies.shift_id')
            ->get()
            ->map(function ($daily) {
                $details = $daily->detail;
                $total = $details->count();

                if ($total === 0) {
                    $daily->status_checklist = 'N/A';
                    return $daily;
                }

                // skor soal
                $totalScore = $details->sum(function ($d) {
                    switch ($d->status_id) {
                        case 1: return 4;
                        case 2: return 3;
                        case 3: return 2;
                        case 4: return 1;
                        default: return 0;
                    }
                });

                // Nilai maksimal = jumlah soal × 4
                $maxScore = $total * 4;
                $persen = ($totalScore / $maxScore) * 100;

                // persentase
                if ($persen >= 80) {
                    $daily->status_checklist = 'Baik';
                } elseif ($persen >= 50) {
                    $daily->status_checklist = 'Cukup';
                } elseif ($persen >= 30) {
                    $daily->status_checklist = 'Rusak';
                } else {
                    $daily->status_checklist = 'N/A';
                }

                return $daily;
            });

        $groupedBySiteShift = $dailyList->groupBy(function ($item) {
            return $item->unit->id . "_" . $item->shift_id ?? 'Unknown Site';
        })->map(fn($rows) => $rows->first());

        return view('dashboard', compact('equipmentStats', 'groupedBySiteShift', "sites"));
    }
}