<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Interfaces\DailyRepositoryInterface;
use App\Interfaces\LocationRepositoryInterface;
use App\Interfaces\ShiftRepositoryInterface;
use App\Interfaces\StatusRepositoryInterface;
use App\Interfaces\UnitRepositoryInterface;
use App\Models\Daily;
use App\Models\DetailDaily;
use App\Models\Location;
use App\Models\Shift;
use Carbon\Carbon;
use Illuminate\Database\QueryException;
use Illuminate\Http\Request;
use PermissionHelper;
use Yajra\DataTables\Facades\DataTables;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;
use ZipArchive;

class DailyController extends Controller
{
    protected $shift;
    protected $unit;
    protected $status;
    protected $location;
    protected $daily;

    public function __construct(
        ShiftRepositoryInterface $shift,
        StatusRepositoryInterface $status,
        UnitRepositoryInterface $unit,
        LocationRepositoryInterface $location,
        DailyRepositoryInterface $daily
    ) {
        $this->shift = $shift;
        $this->unit = $unit;
        $this->status = $status;
        $this->location = $location;
        $this->daily = $daily;
    }
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $canCreate = PermissionHelper::hasPermission("daily-report")->create;
        $option = request()->option;
        $shift = $this->shift->all();
        $location = $this->location->all();
        $unit = $this->unit->all();

        $sitesId = request()->location_id;
        $shiftId = request()->shift_id;
        $date = request()->tanggal;


        $daily = $this->daily->datatable($sitesId, $shiftId, $date);
        if (request()->ajax()) {
            return $this->datatable($daily);
        }
        return view('daily.index', compact('shift', 'unit', 'location', 'canCreate'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'shift_id' => 'required',
            'unit_id' => 'required',
            'form_id' => 'required',
        ]);
        try {
            $unit = $this->unit->find($request->unit_id);
            $location = $this->location->find($unit->location_id);
            $shift = $this->shift->find($request->shift_id);

            $data_ready = Daily::with('shift', 'unit')
                ->where('form_id', $request->form_id)
                ->where('shift_id', $request->shift_id)
                ->where('location_id', $unit->location_id)
                ->where('unit_id', $unit->id)
                ->whereDay('date', now()->day)
                ->whereMonth('date', now()->month)
                ->whereYear('date', now()->year)
                ->first();

            if ($data_ready) {
                $shift = $data_ready->shift->name;
                $unit = $data_ready->unit->unit_code;
                return response()->json([
                    'status' => 2,
                    'message' => 'Mohon maaf, Shift-' . $shift . ', Unit ' . $unit . ' sudah melakukan Checklist Harian.'
                ]);
            }

            $code_report = 'do_' . now()->format('Ymd') . '_' .
                $location->name . '_' .
                $unit->unit_code . '_' .
                $shift->name . '_' .
                str_replace(" ", "_", auth()->user()->name);

            $daily = $this->daily->create([
                "form_id" => $request->form_id,
                "user_id" => auth()->user()->id,
                "shift_id" => $request->shift_id,
                "location_id" => $unit->location_id,
                "unit_id" => $request->unit_id,
                "code_report" => $code_report,
                "date" => now(),
                "status_daily" => 'pending',
                "created_at" => now()
            ]);

            return response()->json([
                'status' => 1,
                'message' => 'Data daily berhasil ditambahkan.',
                'redirect' => route('daily-report.edit', ['daily_report' => $daily->id])
            ]);
        } catch (\Throwable $th) {
            return response()->json([
                'status' => 2,
                'message' => 'Data daily gagal ditambahkan.  ' . $th->getMessage()
            ]);
        }

    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        try {
            $daily = $this->daily->find($id);
            $isPreview = request()->isPreview;
            $year = \Carbon\Carbon::parse($daily->date)->format('Y');
            $month = \Carbon\Carbon::parse($daily->date)->format('m');
            $fileName = "{$daily->code_report}.pdf";
            $shift = Shift::find($daily->shift_id);
            $locationName = str_replace(' ', '_', $daily->location->name);

            $filePath = public_path("PDF_PMM/{$year}/{$month}/{$locationName}/{$shift->name}/{$fileName}");

            if (!File::exists($filePath)) {
                return back()->with('error', 'File PDF tidak ditemukan di folder public.');
            }
            if (ob_get_level()) {
                ob_end_clean();
            }

            if (!$isPreview) {
                return response()->download($filePath);
            } else {
                return response()->file($filePath);
            }


        } catch (\Throwable $th) {
            return back()->with('error', 'Gagal download PDF: ' . $th->getMessage());
        }
    }

    public function downloadAll(Request $request)
    {
        try {
            $siteId = $request->site_id;
            $shiftId = $request->shift_id;
            $tanggal = \Illuminate\Support\Carbon::parse($request->tanggal) ?? date('Y-m-d');

            // Validasi parameter
            if ((!$siteId || $siteId == "all") || (!$shiftId || $shiftId == "all")) {
                return response()->json([
                    'error' => 'Parameter site dan shift wajib diisi!'
                ], 400);
            }

            $year = $tanggal->format("Y");
            $month = $tanggal->format("m");

            $location = Location::find($siteId);
            $shift = Shift::find($shiftId);

            $shiftName = str_replace(' ', '_', $shift->name);
            $locationName = str_replace(' ', '_', $location->name);
            $folderPath = public_path("PDF_PMM/{$year}/{$month}/{$locationName}/{$shiftName}");

            if (!File::exists($folderPath)) {
                return response()->json([
                    'error' => 'Folder tidak ditemukan di public.'
                ], 404);
            }

            // Ambil query dari datatable, lalu jalankan
            $query = $this->daily->datatable($siteId, $shiftId, $tanggal);
            $daily = $query->get(['code_report']); // hanya ambil kolom code_report

            $reportCodes = $daily->pluck('code_report')->filter()->toArray();

            if (empty($reportCodes)) {
                return response()->json([
                    'error' => 'Tidak ada data code_report ditemukan.'
                ], 404);
            }

            // Ambil semua file PDF di folder
            $allFiles = File::files($folderPath);

            // Filter file berdasarkan code_report
            $files = collect($allFiles)->filter(function ($file) use ($reportCodes) {
                $name = pathinfo($file->getFilename(), PATHINFO_FILENAME);
                return in_array($name, $reportCodes);
            });

            if ($files->isEmpty()) {
                return response()->json([
                    'error' => 'Tidak ada file PDF yang cocok dengan code_report.'
                ], 404);
            }

            // Nama file ZIP
            $zipFileName = "do_" . $tanggal->toDateString() . "_{$locationName}_{$shiftName}.zip";
            $zipPath = storage_path("app/public/{$zipFileName}");

            // Hapus jika sudah ada
            if (File::exists($zipPath)) {
                File::delete($zipPath);
            }

            // Buat ZIP
            $zip = new ZipArchive();
            if ($zip->open($zipPath, ZipArchive::CREATE) === true) {
                foreach ($files as $file) {
                    $zip->addFile($file->getPathname(), $file->getFilename());
                }
                $zip->close();
            } else {
                return response()->json([
                    'error' => 'Gagal membuat file ZIP.'
                ], 500);
            }

            return response()->download($zipPath)->deleteFileAfterSend(true);

        } catch (\Throwable $th) {
            return response()->json([
                'error' => 'Terjadi kesalahan: ' . $th->getMessage(),
            ], 500);
        }
    }



    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        $daily = Daily::with('form.sections', 'form.sections.items', 'detail')->where('id', $id)->first();
        if (!$daily) {
            return redirect()->back()->with('error', 'data daily tidak ditemukan.');
        }
        $daily_item = DetailDaily::where('dailies_id', $daily->id)->latest()->get();
        $status = $this->status->all();
        return view('daily.edit', compact('daily', 'status'));
    }


    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {

        $formsItemsId = request()->forms_items_id;
        $daily = $this->daily->find($id);
        $location = $this->location->find($daily->location_id);
        $shift = $this->shift->find($daily->shift_id);
        $unit = $this->unit->find($daily->unit_id);

        foreach ($formsItemsId as $index => $fiList) {
            foreach ($fiList as $i => $fi) {
                DetailDaily::create([
                    "dailies_id" => $id,
                    "form_item_id" => $fi,
                    "status_id" => request()->status[$index][$i] ?? null,
                    "note" => request()->catatan[$index][$i] ?? null,
                    "unit_id" => $daily->unit_id
                ]);
            }
        }
        $this->daily->update($id, ["comment" => request()->downtime_note]);

        $daily = $this->daily->find($id);
        $daily_item = DetailDaily::where('dailies_id', $daily->id)->latest()->get();

        $year = now()->format('Y');
        $month = now()->format('m');
        $locationName = str_replace(" ", "_", $location->name);
        $folderPath = public_path("PDF_PMM/{$year}/{$month}/{$locationName}/{$shift->name}");

        if (!File::exists($folderPath)) {
            File::makeDirectory($folderPath, 0777, true, true);
        }

        $status = $this->status->all();
        $fileName = "{$daily->code_report}.pdf";
        $filePath = "{$folderPath}/{$fileName}";

        $pdf = Pdf::loadView('daily.pdf.index', [
            'daily' => $daily,
            'unit' => $unit,
            'location' => $location,
            'shift' => $shift,
            'daily_item' => $daily_item,
            'status' => $status,
            'user' => auth()->user(),
        ]);

        $pdf->save($filePath);

        $daily->update([
            "status_daily" => "completed"
        ]);

        return redirect()->route('daily-report.index')->with('success', 'Daily Checklist Tersimpan');
        ;

    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        try {
            $this->daily->delete($id);

            return response()->json([
                'success' => true,
                'message' => 'Daily report berhasil dihapus!'
            ]);
        } catch (QueryException $e) {
            if ($e->getCode() == '23000') {
                return response()->json([
                    'success' => false,
                    'message' => 'Gagal menghapus Daily report karena masih digunakan di data lain.'
                ], 400);
            }

            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan saat menghapus Daily report.'
            ], 500);
        }
    }

    public function datatable($data)
    {
        return DataTables::of($data)
            ->addIndexColumn()
            ->addColumn("vehicle", function ($item) {
                return $item->unit->unit_code . ' - ' . $item->unit->unittype->name;
            })
            ->addColumn("Checklist", function ($item) {
                return $item->form->title;
            })
            ->addColumn("user", function ($item) {
                return $item->user->name ?? "-";
            })
            ->addColumn("location", function ($item) {
                return $item->location->name ?? "-";
            })
            ->addColumn("shift", function ($item) {
                return $item->shift->name ?? "-";
            })
            ->addColumn("rusak_count", function ($item) {
                $count = $item->detail()
                    ->whereHas('status', function ($q) {
                        $q->where('name', 'like', '%rusak%');
                    })
                    ->count();
                return $count;
            })
            ->addColumn("status_checklist", function ($item) {
                // Hitung jumlah daily_items dengan status mengandung 'rusak'
                $count = $item->detail()
                    ->whereHas('status', function ($q) {
                    $q->where('name', 'like', '%rusak%');
                })
                    ->count();
                if ($count > 0) {
                    return '<span class="badge bg-danger">Item Rusak</span>';
                } else {
                    return '<span class="badge bg-primary">Tidak Ada Yang Rusak</span>';
                }
            })
            ->addColumn('action', function ($row) {
                $canDelete = PermissionHelper::hasPermission("daily-report")->delete;

                $actionBtn = "<div class='d-flex gap-2'>";
                $actionBtn .= '<a href="javascript:void(0)"
                        data-id="' . $row->id . '"
                        class="pdf btn btn-info btn-sm"><i class="fa fa-print"></i></a>';

                // $actionBtn .= request()->option == "active" ?
                //     '<a href="javascript:void(0)"
                //                 data-id="' . $row->id . '"
                //                 data-name="' . $row->name . '"
                //                 class="edit btn btn-warning btn-sm"><i class="fa fa-pencil"></i></a>' : "";
    
                $actionBtn .= $canDelete ? '<a href="javascript:void(0)"
                                data-id="' . $row->id . '"
                                class="delete btn btn-danger btn-sm"><i class="fa fa-trash"/></i></a>' : "";

                $actionBtn .= "</div>";
                return $actionBtn;
            })

            ->orderColumn('vehicle', function ($query, $order) {
                $query->join('units', 'units.id', '=', 'dailies.unit_id')
                    ->orderBy('units.unit_code', $order);
            })
            ->orderColumn('checklist', function ($query, $order) {
                $query->join('forms', 'forms.id', '=', 'dailies.form_id')
                    ->orderBy('forms.title', $order);
            })
            ->orderColumn('user', function ($query, $order) {
                $query->join('users', 'users.id', '=', 'dailies.user_id')
                    ->orderBy('users.name', $order);
            })
            ->orderColumn('location', function ($query, $order) {
                $query->join('locations', 'locations.id', '=', 'dailies.location_id')
                    ->orderBy('locations.name', $order);
            })
            ->orderColumn('shift', function ($query, $order) {
                $query->join('shifts', 'shifts.id', '=', 'dailies.shift_id')
                    ->orderBy('shifts.name', $order);
            })

            // FILTER
            ->filterColumn('vehicle', function ($query, $keyword) {
                $query->whereHas('unit', function ($q) use ($keyword) {
                    $q->where('unit_code', 'like', "%{$keyword}%")
                        ->orWhereHas('unittype', function ($u) use ($keyword) {
                            $u->where('name', 'like', "%{$keyword}%");
                        });
                });
            })
            ->filterColumn('checklist', function ($query, $keyword) {
                $query->whereHas('form', function ($q) use ($keyword) {
                    $q->where('title', 'like', "%{$keyword}%");
                });
            })
            ->filterColumn('user', function ($query, $keyword) {
                $query->whereHas('user', function ($q) use ($keyword) {
                    $q->where('name', 'like', "%{$keyword}%");
                });
            })
            ->filterColumn('location', function ($query, $keyword) {
                $query->whereHas('location', function ($q) use ($keyword) {
                    $q->where('name', 'like', "%{$keyword}%");
                });
            })
            ->filterColumn('shift', function ($query, $keyword) {
                $query->whereHas('shift', function ($q) use ($keyword) {
                    $q->where('name', 'like', "%{$keyword}%");
                });
            })
            ->rawColumns(['action', 'status_checklist'])
            ->make(true);
    }
}
