<?php

namespace App\Http\Controllers\Api;

use App\Interfaces\LocationRepositoryInterface;
use App\Interfaces\StbmRepositoryInterface;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use PermissionHelper;
use Yajra\DataTables\Facades\DataTables;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Log;

class StbmController extends Controller
{

    protected $stbm;
    protected $location;
    public function __construct(StbmRepositoryInterface $stbm, LocationRepositoryInterface $locationRepository)
    {
        $this->stbm = $stbm;
        $this->location = $locationRepository;
    }
    public function index()
    {
        $search = request()->query('search');
        $data = $this->stbm->datatable()
            ->where("topic", "LIKE", "%{$search}%")
            ->orWhere("description", "LIKE", "%{$search}%")
            ->orWhere("date", "LIKE", "%{$search}%")
            ->orWhereHas("location", function ($rw) use ($search) {
                $rw->where("name", "LIKE", "%{$search}%");
            })
            ->orWhereHas("user", function ($rw) use ($search) {
                $rw->where("name", "LIKE", "%{$search}%");
            })
            ->get()
            ->map(function ($post) {
                return [
                    'id' => $post->id,
                    'name' => $post->user->name,
                    'topic' => $post->topic,
                    'location' => $post->location->name,
                    'date' => $post->date
                ];
            });
        ;
        return response()->json([
            'status' => 'success',
            'data' => $data
        ]);
    }

    public function store(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'location_id' => 'required|integer|exists:locations,id',
            'date' => 'required|date',
            'topic' => 'required|string|max:255',
            'description' => 'required|string',
            'images.*' => 'nullable|image|mimes:jpg,jpeg,png,gif,webp|max:2048',
            'participant' => 'nullable|array',
            'participant.*' => 'nullable|string|max:100',
        ]);

        $validator->after(function ($validator) use ($request) {
            $participants = collect($request->participant)->filter(fn($p) => !empty(trim($p)));
            $images = $request->hasFile('images');

            if ($participants->count() < 1) {
                $validator->errors()->add('participant[0]', 'Minimal 1 peserta harus diisi.');
            }
            if (!$images) {
                $validator->errors()->add('images[0]', 'Minimal 1 dokumentasi harus diisi.');
            }
        });

        if ($validator->fails()) {
            Log::error('STBMI Validation Error:', $validator->errors()->toArray());
            Log::error('Request Data:', $request->all());
            return response()->json([
                'status' => 'error',
                'message' => 'Validasi gagal',
                'errors' => $validator->errors(),
                "requests" => request()->all()
            ], 422);
        }

        $validated = $validator->validated();


        try {

            $result = $this->stbm->create(request()->all());
            return response()->json([
                'success' => true,
                'message' => 'STBM berhasil dibuat',
                'data' => $result,
            ], 201);
        } catch (\Throwable $th) {
            Log::error('STBM Exception: ' . $th->getMessage());
            Log::error($th->getTraceAsString());
            return response()->json([
                'success' => false,
                'message' => 'Gagal menyimpan STBM ' . $th->getMessage(),
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        try {
            $stbm = $this->stbm->find($id);
            return response()->json([
                "status" => "success",
                "data" => [
                    'id' => $stbm->id,
                    'name' => $stbm->user->name,
                    'topic' => $stbm->topic,
                    'description' => $stbm->description,
                    'location' => $stbm->location->name,
                    'date' => $stbm->date,
                    'images' => $stbm->images()->select("image")->get()->map(fn($img) => asset('storage/' . $img->image)),
                    'participant' => $stbm->participant()->select("name")->get(),
                ]
            ], 200);
        } catch (\Throwable $th) {
            return response()->json(["message" => "Terjadi kesalahan dengan server"], 500);
        }
    }

    public function donwloadPdf($id)
    {
        try {
            $stbm = $this->stbm->find($id);
            $pdf = Pdf::loadView('stbm.pdf', [
                "stbm" => $stbm
            ])
                ->setPaper('a4', 'portrait');

            return $pdf->download('STBM' . '.pdf');

        } catch (Exception $e) {
            return response()->json(['message' => 'Data laporan ini kosong atau tidak ada checklist'], 404);
        }
    }

    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'location_id' => 'required|integer|exists:locations,id',
            'date' => 'required|date',
            'topic' => 'required|string|max:255',
            'description' => 'required|string',
            'images.*' => 'nullable|image|mimes:jpg,jpeg,png,gif,webp|max:2048',
            'participant' => 'nullable|array',
            'participant.*' => 'nullable|string|max:100',
        ]);

        $validator->after(function ($validator) use ($request) {
            $participants = collect($request->participant)
                ->filter(fn($p) => !empty(trim($p))); // ambil yg tidak kosong

            $images = $request->hasFile('images') || count($request->old_images ?? []) > 0;

            if ($participants->count() < 1) {
                $validator->errors()->add('participant[0]', 'Minimal 1 peserta harus diisi.');
            }

            if (!$images) {
                $validator->errors()->add('images[0]', 'Minimal 1 dokumentasi harus diisi.');
            }
        });

        if ($validator->fails()) {
            Log::error('STBMU Validation Error:', $validator->errors()->toArray());
            Log::error('Request Data:', $request->all());
            return response()->json([
                'status' => 'error',
                'message' => 'Validasi gagal',
                'errors' => $validator->errors(),
            ], 422);
        }

        $validated = $validator->validated();


        try {

            $result = $this->stbm->update($id, request()->all(), true);
            return response()->json([
                'success' => true,
                'message' => 'STBM berhasil diubah',
                'data' => $result,
            ], 201);
        } catch (\Throwable $th) {
            Log::error('STBM Update Exception: ' . $th->getMessage());
            Log::error($th->getTraceAsString());
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengubah STBM ' . $th->getMessage(),
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        try {
            $this->stbm->delete($id);
            return response()->json([
                'success' => true,
                'message' => 'STBM berhasil dihapus',
            ], 200);
        } catch (\Throwable $th) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal menghapus STBM ' . $th->getMessage(),
            ], 500);
        }
    }

    public function getLocation()
    {
        $data = $this->location->all();

        return response()->json([
            'message' => 'success',
            'data' => $data
        ]);
    }


}
